/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Booleans;
import java.math.BigInteger;
import java.math.RoundingMode;

public final class DoubleMath {
    private static final double MIN_INT_AS_DOUBLE = -2.147483648E9;
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final double LN_2 = Math.log(2.0);
    @VisibleForTesting
    static final int MAX_FACTORIAL = 170;
    @VisibleForTesting
    static final double[] everySixteenthFactorial = new double[]{1.0, 2.0922789888E13, 2.631308369336935E35, 1.2413915592536073E61, 1.2688693218588417E89, 7.156945704626381E118, 9.916779348709496E149, 1.974506857221074E182, 3.856204823625804E215, 5.5502938327393044E249, 4.7147236359920616E284};

    static double roundIntermediate(double x, RoundingMode mode2) {
        if (!DoubleUtils.isFinite(x)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (mode2) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isMathematicalInteger(x));
                return x;
            }
            case FLOOR: {
                if (x >= 0.0 || DoubleMath.isMathematicalInteger(x)) {
                    return x;
                }
                return x - 1.0;
            }
            case CEILING: {
                if (x <= 0.0 || DoubleMath.isMathematicalInteger(x)) {
                    return x;
                }
                return x + 1.0;
            }
            case DOWN: {
                return x;
            }
            case UP: {
                if (DoubleMath.isMathematicalInteger(x)) {
                    return x;
                }
                return x + Math.copySign(1.0, x);
            }
            case HALF_EVEN: {
                return Math.rint(x);
            }
            case HALF_UP: {
                double z2 = Math.rint(x);
                if (Math.abs(x - z2) == 0.5) {
                    return x + Math.copySign(0.5, x);
                }
                return z2;
            }
            case HALF_DOWN: {
                double z3 = Math.rint(x);
                if (Math.abs(x - z3) == 0.5) {
                    return x;
                }
                return z3;
            }
        }
        throw new AssertionError();
    }

    public static int roundToInt(double x, RoundingMode mode2) {
        double z2 = DoubleMath.roundIntermediate(x, mode2);
        MathPreconditions.checkInRange(z2 > -2.147483649E9 & z2 < 2.147483648E9);
        return (int)z2;
    }

    public static long roundToLong(double x, RoundingMode mode2) {
        double z2 = DoubleMath.roundIntermediate(x, mode2);
        MathPreconditions.checkInRange(-9.223372036854776E18 - z2 < 1.0 & z2 < 9.223372036854776E18);
        return (long)z2;
    }

    public static BigInteger roundToBigInteger(double x, RoundingMode mode2) {
        if (-9.223372036854776E18 - (x = DoubleMath.roundIntermediate(x, mode2)) < 1.0 & x < 9.223372036854776E18) {
            return BigInteger.valueOf((long)x);
        }
        int exponent = Math.getExponent(x);
        long significand = DoubleUtils.getSignificand(x);
        BigInteger result2 = BigInteger.valueOf(significand).shiftLeft(exponent - 52);
        return x < 0.0 ? result2.negate() : result2;
    }

    public static boolean isPowerOfTwo(double x) {
        return x > 0.0 && DoubleUtils.isFinite(x) && LongMath.isPowerOfTwo(DoubleUtils.getSignificand(x));
    }

    public static double log2(double x) {
        return Math.log(x) / LN_2;
    }

    public static int log2(double x, RoundingMode mode2) {
        boolean increment2;
        Preconditions.checkArgument(x > 0.0 && DoubleUtils.isFinite(x), "x must be positive and finite");
        int exponent = Math.getExponent(x);
        if (!DoubleUtils.isNormal(x)) {
            return DoubleMath.log2(x * 4.503599627370496E15, mode2) - 52;
        }
        switch (mode2) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isPowerOfTwo(x));
            }
            case FLOOR: {
                increment2 = false;
                break;
            }
            case CEILING: {
                increment2 = !DoubleMath.isPowerOfTwo(x);
                break;
            }
            case DOWN: {
                increment2 = exponent < 0 & !DoubleMath.isPowerOfTwo(x);
                break;
            }
            case UP: {
                increment2 = exponent >= 0 & !DoubleMath.isPowerOfTwo(x);
                break;
            }
            case HALF_EVEN: 
            case HALF_UP: 
            case HALF_DOWN: {
                double xScaled = DoubleUtils.scaleNormalize(x);
                increment2 = xScaled * xScaled > 2.0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment2 ? exponent + 1 : exponent;
    }

    public static boolean isMathematicalInteger(double x) {
        return DoubleUtils.isFinite(x) && (x == 0.0 || 52 - Long.numberOfTrailingZeros(DoubleUtils.getSignificand(x)) <= Math.getExponent(x));
    }

    public static double factorial(int n) {
        MathPreconditions.checkNonNegative("n", n);
        if (n > 170) {
            return Double.POSITIVE_INFINITY;
        }
        double accum = 1.0;
        for (int i = 1 + (n & 0xFFFFFFF0); i <= n; ++i) {
            accum *= (double)i;
        }
        return accum * everySixteenthFactorial[n >> 4];
    }

    public static boolean fuzzyEquals(double a2, double b2, double tolerance) {
        MathPreconditions.checkNonNegative("tolerance", tolerance);
        return Math.copySign(a2 - b2, 1.0) <= tolerance || a2 == b2 || a2 != a2 && b2 != b2;
    }

    public static int fuzzyCompare(double a2, double b2, double tolerance) {
        if (DoubleMath.fuzzyEquals(a2, b2, tolerance)) {
            return 0;
        }
        if (a2 < b2) {
            return -1;
        }
        if (a2 > b2) {
            return 1;
        }
        return Booleans.compare(Double.isNaN(a2), Double.isNaN(b2));
    }

    private DoubleMath() {
    }
}

